/*
 * Decompiled with CFR 0.152.
 */
package io.seata.server.session;

import io.seata.common.util.CollectionUtils;
import io.seata.config.Configuration;
import io.seata.config.ConfigurationFactory;
import io.seata.core.exception.TransactionException;
import io.seata.core.model.BranchType;
import io.seata.core.model.GlobalStatus;
import io.seata.server.UUIDGenerator;
import io.seata.server.coordinator.DefaultCoordinator;
import io.seata.server.metrics.MetricsPublisher;
import io.seata.server.session.BranchSession;
import io.seata.server.session.BranchSessionHandler;
import io.seata.server.session.GlobalSession;
import io.seata.server.session.GlobalSessionHandler;
import io.seata.server.session.SessionStatusValidator;
import io.seata.server.store.StoreConfig;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class SessionHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(SessionHelper.class);
    private static final Configuration CONFIG = ConfigurationFactory.getInstance();
    private static final Boolean ENABLE_BRANCH_ASYNC_REMOVE = CONFIG.getBoolean("server.session.enableBranchAsyncRemove", false);
    private static final DefaultCoordinator COORDINATOR = DefaultCoordinator.getInstance();
    private static final boolean DELAY_HANDLE_SESSION = StoreConfig.getSessionMode() != StoreConfig.SessionMode.FILE;

    private SessionHelper() {
    }

    public static BranchSession newBranchByGlobal(GlobalSession globalSession, BranchType branchType, String resourceId, String lockKeys, String clientId) {
        return SessionHelper.newBranchByGlobal(globalSession, branchType, resourceId, null, lockKeys, clientId);
    }

    public static BranchSession newBranchByGlobal(GlobalSession globalSession, BranchType branchType, String resourceId, String applicationData, String lockKeys, String clientId) {
        BranchSession branchSession = new BranchSession();
        branchSession.setXid(globalSession.getXid());
        branchSession.setTransactionId(globalSession.getTransactionId());
        branchSession.setBranchId(UUIDGenerator.generateUUID());
        branchSession.setBranchType(branchType);
        branchSession.setResourceId(resourceId);
        branchSession.setLockKey(lockKeys);
        branchSession.setClientId(clientId);
        branchSession.setApplicationData(applicationData);
        return branchSession;
    }

    public static BranchSession newBranch(BranchType branchType, String xid, long branchId, String resourceId, String applicationData) {
        BranchSession branchSession = new BranchSession();
        branchSession.setXid(xid);
        branchSession.setBranchId(branchId);
        branchSession.setBranchType(branchType);
        branchSession.setResourceId(resourceId);
        branchSession.setApplicationData(applicationData);
        return branchSession;
    }

    public static void endCommitted(GlobalSession globalSession, boolean retryGlobal) throws TransactionException {
        if (retryGlobal || !DELAY_HANDLE_SESSION) {
            long beginTime = System.currentTimeMillis();
            boolean retryBranch = globalSession.getStatus() == GlobalStatus.CommitRetrying;
            globalSession.changeGlobalStatus(GlobalStatus.Committed);
            globalSession.end();
            if (!DELAY_HANDLE_SESSION) {
                MetricsPublisher.postSessionDoneEvent(globalSession, false, false);
            }
            MetricsPublisher.postSessionDoneEvent(globalSession, "AfterCommitted", true, beginTime, retryBranch);
        } else {
            MetricsPublisher.postSessionDoneEvent(globalSession, false, false);
        }
    }

    public static void endCommitFailed(GlobalSession globalSession, boolean retryGlobal) throws TransactionException {
        SessionHelper.endCommitFailed(globalSession, retryGlobal, false);
    }

    public static void endCommitFailed(GlobalSession globalSession, boolean retryGlobal, boolean isRetryTimeout) throws TransactionException {
        if (isRetryTimeout) {
            globalSession.changeGlobalStatus(GlobalStatus.CommitRetryTimeout);
        } else {
            globalSession.changeGlobalStatus(GlobalStatus.CommitFailed);
        }
        LOGGER.error("The Global session {} has changed the status to {}, need to be handled it manually.", (Object)globalSession.getXid(), (Object)globalSession.getStatus());
        globalSession.end();
        MetricsPublisher.postSessionDoneEvent(globalSession, retryGlobal, false);
    }

    public static void endRollbacked(GlobalSession globalSession, boolean retryGlobal) throws TransactionException {
        if (retryGlobal || !DELAY_HANDLE_SESSION) {
            boolean retryBranch;
            long beginTime = System.currentTimeMillis();
            boolean timeoutDone = false;
            GlobalStatus currentStatus = globalSession.getStatus();
            if (currentStatus == GlobalStatus.TimeoutRollbacking) {
                MetricsPublisher.postSessionDoneEvent(globalSession, GlobalStatus.TimeoutRollbacked, false, false);
                timeoutDone = true;
            }
            boolean bl = retryBranch = currentStatus == GlobalStatus.TimeoutRollbackRetrying || currentStatus == GlobalStatus.RollbackRetrying;
            if (SessionStatusValidator.isTimeoutGlobalStatus(currentStatus)) {
                globalSession.changeGlobalStatus(GlobalStatus.TimeoutRollbacked);
            } else {
                globalSession.changeGlobalStatus(GlobalStatus.Rollbacked);
            }
            globalSession.end();
            if (!DELAY_HANDLE_SESSION && !timeoutDone) {
                MetricsPublisher.postSessionDoneEvent(globalSession, false, false);
            }
            MetricsPublisher.postSessionDoneEvent(globalSession, "AfterRollbacked", true, beginTime, retryBranch);
        } else {
            MetricsPublisher.postSessionDoneEvent(globalSession, GlobalStatus.Rollbacked, false, false);
        }
    }

    public static void endRollbackFailed(GlobalSession globalSession, boolean retryGlobal) throws TransactionException {
        SessionHelper.endRollbackFailed(globalSession, retryGlobal, false);
    }

    public static void endRollbackFailed(GlobalSession globalSession, boolean retryGlobal, boolean isRetryTimeout) throws TransactionException {
        GlobalStatus currentStatus = globalSession.getStatus();
        if (isRetryTimeout) {
            globalSession.changeGlobalStatus(GlobalStatus.RollbackRetryTimeout);
        } else if (SessionStatusValidator.isTimeoutGlobalStatus(currentStatus)) {
            globalSession.changeGlobalStatus(GlobalStatus.TimeoutRollbackFailed);
        } else {
            globalSession.changeGlobalStatus(GlobalStatus.RollbackFailed);
        }
        LOGGER.error("The Global session {} has changed the status to {}, need to be handled it manually.", (Object)globalSession.getXid(), (Object)globalSession.getStatus());
        globalSession.end();
        MetricsPublisher.postSessionDoneEvent(globalSession, retryGlobal, false);
    }

    public static void forEach(Collection<GlobalSession> sessions, GlobalSessionHandler handler) {
        if (CollectionUtils.isEmpty(sessions)) {
            return;
        }
        sessions.parallelStream().forEach(globalSession -> {
            try {
                MDC.put((String)"X-TX-XID", (String)globalSession.getXid());
                handler.handle((GlobalSession)globalSession);
            }
            catch (Throwable th) {
                LOGGER.error("handle global session failed: {}", (Object)globalSession.getXid(), (Object)th);
            }
            finally {
                MDC.remove((String)"X-TX-XID");
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Boolean forEach(Collection<BranchSession> sessions, BranchSessionHandler handler) throws TransactionException {
        for (BranchSession branchSession : sessions) {
            try {
                MDC.put((String)"X-TX-BRANCH-ID", (String)String.valueOf(branchSession.getBranchId()));
                Boolean result = handler.handle(branchSession);
                if (result == null) continue;
                Boolean bl = result;
                return bl;
            }
            finally {
                MDC.remove((String)"X-TX-BRANCH-ID");
            }
        }
        return null;
    }

    public static void removeBranch(GlobalSession globalSession, BranchSession branchSession, boolean isAsync) throws TransactionException {
        if (Objects.equals(Boolean.TRUE, ENABLE_BRANCH_ASYNC_REMOVE) && isAsync) {
            COORDINATOR.doBranchRemoveAsync(globalSession, branchSession);
        } else {
            globalSession.removeBranch(branchSession);
        }
    }

    public static void removeAllBranch(GlobalSession globalSession, boolean isAsync) throws TransactionException {
        List<BranchSession> branchSessions = globalSession.getSortedBranches();
        if (branchSessions == null || branchSessions.isEmpty()) {
            return;
        }
        if (Objects.equals(Boolean.TRUE, ENABLE_BRANCH_ASYNC_REMOVE) && isAsync) {
            COORDINATOR.doBranchRemoveAllAsync(globalSession);
        } else {
            for (BranchSession branchSession : branchSessions) {
                globalSession.removeBranch(branchSession);
            }
        }
    }
}

